;(function(){
    'use strict';

    /**
     * send jPList redraw event
     * @param {Object} context
     */
    var sendEvent = function(context){

        var status = getStatus(context, false);

        //update last status
        context.history.addStatus(status);

        context.observer.trigger(context.observer.events.knownStatusesChanged, [[status]]);
    };

    /**
     * handle place change
     * @param {Object} context
     */
    var handlePlaceChange = function(context){

        //get selected place
        context.params.place = context.params.autocomplete.getPlace();

        //send jPList redraw event
        sendEvent(context);
    };

    /**
     * reset autocomplete
     * @param {Object} context
     */
    var resetAutocomplete = function(context){

        context.params.$input.val('');
        context.params.place = null;
    };

    /**
     * init events
     * @param {Object} context
     */
    var initEvents = function(context){

        if(context.params.handler){

            window.google.maps.event.removeListener(context.params.handler);
        }

        /**
         * on autocomplete change
         */
        context.params.handler = window.google.maps.event.addListener(context.params.autocomplete, 'place_changed', function(){

            handlePlaceChange(context);
        });

        /**
         * on autocomplete textbox clear
         */
        context.$control.find('[data-type="clear"]').off('click').on('click', function(){

            //reset autocomplete
            resetAutocomplete(context);

            //send jPList redraw event
            sendEvent(context);
        });

        /**
         * on empty textbox
         */
        context.params.$input.off('keyup').on('keyup', function(e){

            var val;

            //get val
            val = jQuery.trim(jQuery(this).val());

            if(val === ''){

                context.params.place = null;

                //send jPList redraw event
                sendEvent(context);
            }
        });

        /**
         * on radius dropdown change
         */
        context.params.$select.off('change').on('change', function(){
            sendEvent(context);
        });
    };

    /**
     * Set control status
     * @param {Object} context
     * @param {jQuery.fn.jplist.StatusDTO|Array.<jQuery.fn.jplist.StatusDTO>} status
     * @param {boolean} restoredFromStorage - is status restored from storage
     */
    var setStatus = function(context, status, restoredFromStorage){

        if(status && status.data && !status.data.name){

            //reset autocomplete
            resetAutocomplete(context);
        }
    };

    /**
     * Get control status
     * @param {Object} context
     * @param {boolean} isDefault - if true, get default (initial) control status; else - get current control status
     * @return {jQuery.fn.jplist.StatusDTO}
     */
    var getStatus = function(context, isDefault){

        var status = null
            ,data
            ,radius;

        if(isDefault || !(context.params.place)){

            data = new jQuery.fn.jplist.controls.AutocompleteDTO(0 ,0 ,'', 0);
        }
        else{

            if(context.params.$select.length > 0){

                radius = Number($.trim(context.params.$select.val())) || 0;
            }
            else{
                radius = context.params.defaultRadius;
            }

            //change radius value to meters
            if(context.params.radiusUnits === 'km'){

                radius = radius * 1000;
            }
            else{ //miles

                //get distance in meters: 1 Mile = 1609.34 meters
                radius = radius * 1609.34;
            }

            data = new jQuery.fn.jplist.controls.AutocompleteDTO(
                context.params.place.geometry.location['lat']()
                ,context.params.place.geometry.location['lng']()
                ,context.params.place.name
                ,radius
            );
        }

        //create status object
        status = new jQuery.fn.jplist.StatusDTO(
            context.name
            ,context.action
            ,context.type
            ,data
            ,context.inStorage
            ,context.inAnimation
            ,context.isAnimateToTop
            ,context.inDeepLinking
        );

        return status;
    };

    /**
     * Google Maps Autocomplete Control
     * @constructor
     * @param {Object} context
     */
    var Init = function(context){

        context.params = {

            autocomplete: null
            ,place: null
            ,$input: context.$control.find('input[type="text"]')
            ,$select: context.$control.find('select')
            ,defaultRadius: Number(context.$control.attr('data-radius')) || 0
            ,radiusUnits: $.trim(context.$control.attr('data-units')) || 'miles'
        };

        if(window.google && window.google.maps){
            window.google.maps.event.addDomListener(window, 'load', function(){

                //init autocomplete
                context.params.autocomplete = new window.google.maps.places.Autocomplete(context.params.$input.get(0));

                //init events
                initEvents(context);

            });
        }

        return jQuery.extend(this, context);
    };

    /**
     * Get control status
     * @param {boolean} isDefault - if true, get default (initial) control status; else - get current control status
     * @return {jQuery.fn.jplist.StatusDTO}
     */
    Init.prototype.getStatus = function(isDefault){
        return getStatus(this, isDefault);
    };

    /**
     * Set Status
     * @param {jQuery.fn.jplist.StatusDTO} status
     * @param {boolean} restoredFromStorage - is status restored from storage
     */
    Init.prototype.setStatus = function(status, restoredFromStorage){
        setStatus(this, status, restoredFromStorage);
    };

    /**
     * Google Maps Autocomplete Control
     * @constructor
     * @param {Object} context
     */
    jQuery.fn.jplist.controls.GoogleAutocompleteControl = function(context){

        return new Init(context);
    };

    /**
     * control registration in jPList plugin system
     */
    jQuery.fn.jplist.controlTypes['google-autocomplete-control'] = {
        className: 'GoogleAutocompleteControl'
        ,options: {}
    };
})();
